/**
 * \file: svg_bitmap_decoder.c
 *
 * version: $Id: svg_bitmap_decoder.c,v 1.19 2010/01/14 13:43:29 tkniep Exp $
 *
 * This file implements the interface functions of the 
 * Bitmap Decoder
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/

#include <grl_os_abstraction.h>
#include "svg_bitmap_decoder.h"
#include "grl_bitmap_decoder_util.h"
#include "grl_bitmap_decoder_core.h"

#include <stdio.h>
#include <string.h>
/*******************************************************************************************
 *   Global Variables
 *******************************************************************************************/

/**
 * Global Version Information, see svg_bitmap_decoder.h for actual value
 */
const U8 g_SVGBMPDEC_version[] = SVG_BMPDEC_MODULE_VER;

/**
 * Global flag for indicating whether the Bitmap
 * Decoder has been initialized.
 * Access to this flag does not need to be protected because
 * it is only written during startup and shutdown of the
 * bitmap decoder
 */
static SVGBoolean   g_bmpdec_initialized  = SVG_FALSE;

/* Dlt specific variables for ctx and debug level */
void *grl_sbmp_dlt_ctx = NULL;
SVGUint8 grl_sbmp_debugLevel = SVG_LOG_FATAL;

void sbmp_dlt_SetDebugLevel(SVGUint8 Level );
void sbmp_dlt_GetDebugLevel( SVGUint8 *DebugLevel  );
SVGInt32 sbmpDltLogLevelInjection(SVGUint32 service_id, void * data, SVGUint32 length);



/*******************************************************************************
**  sbmp_dlt_SetDebugLevel
**  Set the debug level.
**
**  INPUT:
**      SVGUint32 Level
**          New debug level.
**  OUTPUT:
**      Nothing.
*/

void sbmp_dlt_SetDebugLevel(SVGUint8 Level )
{
	grl_sbmp_debugLevel = Level;
}

/*******************************************************************************
**  sbmp_dlt_GetDebugLevel
**  Get the current debug level.
**
**  INPUT:
**      Nothing.
**  OUTPUT:
**      SVGUint32 DebugLevel
**          Handle to store the debug level.
*/
void sbmp_dlt_GetDebugLevel( SVGUint8 *DebugLevel  )
{
    *DebugLevel = grl_sbmp_debugLevel;
}

SVGInt32 sbmpDltLogLevelInjection(SVGUint32 service_id, void * data,
					SVGUint32 length)
{
	SVGUint32 i = 0;
	char* buf = (char *)data;
	UNUSED_ARG(length);

	SVGUint8 newDebugLevel = SVG_LOG_FATAL;

	sbmp_dlt_GetDebugLevel(&newDebugLevel);

	for (i = 0 ; buf[i]; i++)
		buf[i] = tolower(buf[i]);
	if (strncmp("fatal", buf, 5) == 0)
		newDebugLevel = SVG_LOG_FATAL;
	else if (strncmp("error", buf, 5) == 0)
		newDebugLevel = SVG_LOG_ERROR;
	else if (strncmp("warning", buf, 7) == 0)
		newDebugLevel = SVG_LOG_WARN;
	else if (strncmp("info", buf, 4) == 0)
		newDebugLevel = SVG_LOG_INFO;
	else if (strncmp("debug", buf, 5) == 0)
		newDebugLevel = SVG_LOG_DEBUG;
	else if (strncmp("verbose", buf, 7) == 0)
		newDebugLevel = SVG_LOG_VERBOSE;

	sbmp_dlt_SetDebugLevel(newDebugLevel);

	SVG_BMP_U("Log level set to %u for service ID %x ", newDebugLevel,service_id);

	return 0;
}

/*******************************************************************************************
 *   Interface Function Implementations
 *   (documentation is in header file)
 *******************************************************************************************/

void svgInitBmpDec( void )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    
    if ( SVG_FALSE == g_bmpdec_initialized )
    {        
    	/* Clear global error */
        GRL_BMPDEC_clr_error( NULL );

		grl_sbmp_dlt_ctx = svg_init_dlt("SBMP", "DLT CONTEXT CREATED FOR SVG BITMAP DECODER",
				DLT_SERVICE_LOGLVL_SBMP,
				sbmpDltLogLevelInjection );

		SVG_BMP_U("ENTER INTO SVGINITBMPDEC");

        /* Read configuration from devconf but ignore errors as
         * default values are used if an error occurs here
         */
        ret_err = GRL_BMPDEC_read_conf();

        /* Initialize function tables */
        ret_err = GRL_BMPDEC_init_tables();
        if ( SVG_NO_ERROR != ret_err )
        {
            GRL_BMPDEC_set_error( NULL, ret_err );
        }
        else
        {
            /* Register decoder modules */
            ret_err = GRL_BMPDEC_reg_decoder();
            
            if ( SVG_NO_ERROR != ret_err )
            {
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
        }
        if ( SVG_NO_ERROR == ret_err )
        {
            /* Initialize context list */
            ret_err = GRL_BMPDEC_clst_init();

            if ( SVG_NO_ERROR != ret_err )
            {
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
        }

        if ( SVG_NO_ERROR == ret_err )
        {
            g_bmpdec_initialized = SVG_TRUE;
        }
    }
    else
    {
        ret_err = SVG_ALREADY_IN_USE;
        GRL_BMPDEC_set_error( NULL, ret_err );
        SVG_BMP_W("SVG_ALREADY_IN_USE IN SVGINITBMPDEC");
    }

    SVG_BMP_U("EXIT FROM SVGINITBMPDEC");
}


SVGError svgCloseBmpDec( void )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    SVGError    dsy_err         = SVG_NO_ERROR;
    SVGUint32   ctx_num         = 0;
    SVGUint32   ctx_cnt         = 0;

    SVG_BMP_U("ENTER INTO SVGCLOSEBMPDEC");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        /* Destroy all remaining contexts, if any */
        ctx_num = GRL_BMPDEC_clst_get_num();
        while ( ctx_cnt < ctx_num )
        {
            svgGetBmpDecError( NULL );
            svgDestroyContextBmpDec( GRL_BMPDEC_clst_get_ctx( ctx_cnt ) );
            dsy_err = svgGetBmpDecError( NULL );
            if ( SVG_NO_ERROR != dsy_err )
            {
                ret_err = dsy_err;
            }
            ctx_cnt++;
        }

        GRL_BMPDEC_clst_uninit();

        g_bmpdec_initialized = SVG_FALSE;
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGCLOSEBMPDEC");
    }

    SVG_BMP_U("EXIT FROM SVGCLOSEBMPDEC");
    return ret_err;
}


SVGContextBmpDec* svgCreateContextBmpDec( void )
{
    SVGContextBmpDec    *p_bmp_ctx              = NULL;
    SVGError            ret_err;

    SVG_BMP_U("ENTER INTO SVGCREATECONTEXTBMPDEC");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        ret_err = GRL_BMPDEC_cre_ctx( &p_bmp_ctx );
        if ( SVG_NO_ERROR != ret_err )
        {            
            GRL_BMPDEC_set_error( NULL, ret_err );
            p_bmp_ctx = NULL;
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( NULL, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGCREATECONTEXTBMPDEC");
    }

    SVG_BMP_U("EXIT FROM SVGCREATECONTEXTBMPDEC");
    return  p_bmp_ctx;
}


void svgDestroyContextBmpDec( SVGContextBmpDec*   ctx )
{
    SVGError            ret_err  = SVG_NO_ERROR;
    SVGBoolean          is_valid = SVG_FALSE;

    SVG_BMP_U("ENTER INTO SVGDESTROYCONTEXTBMPDEC");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        if ( ctx == NULL )
        {
            ret_err = SVG_CONTEXT_NULL;
            GRL_BMPDEC_set_error( ctx, ret_err );
            SVG_BMP_E("SVG_CONTEXT_NULL IN SVGDESTROYCONTEXTBMPDEC");
        }
        else
        {
            /* Only destroy valid contexts */
            is_valid = GRL_BMPDEC_clst_is_valid( ctx );
            
            if ( SVG_FALSE == is_valid )
            {
                ret_err = SVG_BMPDEC_INVALID_CONTEXT;
                SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN SVGDESTROYCONTEXTBMPDEC");
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
            else
            {
                ret_err = GRL_BMPDEC_dty_ctx( ctx );
                if ( SVG_NO_ERROR != ret_err )
                {
                    GRL_BMPDEC_set_error( NULL, ret_err );
                }                
            }
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( NULL, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGDESTROYCONTEXTBMPDEC");
    }

    SVG_BMP_U("EXIT FROM SVGDESTROYCONTEXTBMPDEC");
}


void svgGetImageInfo( SVGContextBmpDec  *ctx,
                      const SVGImage    *image,
                      SVGImageInfo      *image_info )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    SVGBoolean  is_valid        = SVG_TRUE;

    SVG_BMP_U("ENTER INTO SVGGETIMAGEINFO");
    
    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        /* Check params */
        if ( (ctx == NULL) || (image == NULL)  || (image_info == NULL) )
        {
            if (ctx == NULL)
            {
                /* Context NULL */
                ret_err = SVG_CONTEXT_NULL;
                GRL_BMPDEC_set_error( ctx, ret_err );
                SVG_BMP_E("SVG_CONTEXT_NULL IN SVGGETIMAGEINFO");
            }
            else
            {
                /* Image or Image info pointer is NULL */
                ret_err = SVG_POINTER_NULL;
                GRL_BMPDEC_set_error( ctx, ret_err );
                SVG_BMP_E("SVG_POINTER_NULL IN SVGGETIMAGEINFO");
            }
        }
        else
        {
            /* Only accept valid contexts */
            is_valid = GRL_BMPDEC_clst_is_valid( ctx );
            
            if ( SVG_FALSE == is_valid )
            {
                ret_err = SVG_BMPDEC_INVALID_CONTEXT;
                SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN SVGGETIMAGEINFO");
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
            else
            {
                ret_err = GRL_BMPDEC_check_img_basic( image );
                if ( SVG_NO_ERROR != ret_err )
                {
                    GRL_BMPDEC_set_error( ctx, ret_err );
                    SVG_BMP_E("SVGGETIMAGEINFO:Errors in supplied image structure ");
                }
                else
                {
                    ret_err = GRL_BMPDEC_get_image_info( ctx, image, image_info );
                    if ( SVG_NO_ERROR != ret_err )
                    {
                        GRL_BMPDEC_set_error( ctx, ret_err );
                    }
                }
            }
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( ctx, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGGETIMAGEINFO");
    }

    SVG_BMP_U("EXIT FROM SVGGETIMAGEINFO");
}


void svgGetBmpDecFormats( SVGContextBmpDec       *ctx,
                          SVGBmpDecFormats       *formats )
{
    SVGError            ret_err         = SVG_NO_ERROR;
    SVGBoolean          is_valid        = SVG_FALSE;
    
    SVG_BMP_U("ENTER INTO SVGGETBMPDECFORMATS");
    
    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        /* Check params */
        if ( (ctx == NULL) || (formats == NULL) )
        {
            if (ctx == NULL)
            {
                /* Context NULL */
                ret_err = SVG_CONTEXT_NULL;
                GRL_BMPDEC_set_error( ctx, ret_err );
                SVG_BMP_E("SVG_CONTEXT_NULL IN SVGGETBMPDECFORMATS");
            }
            else
            {
                /* Format pointer is NULL */
                ret_err = SVG_POINTER_NULL;
                GRL_BMPDEC_set_error( ctx, ret_err );
                SVG_BMP_E("SVG_POINTER_NULL IN SVGGETBMPDECFORMATS");
            }
        }
        else
        {
            /* Only handle valid contexts */
            is_valid = GRL_BMPDEC_clst_is_valid( ctx );
            
            if ( SVG_FALSE == is_valid )
            {
                ret_err = SVG_BMPDEC_INVALID_CONTEXT;
                SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN SVGGETBMPDECFORMATS");
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
            else
            {
                ret_err = GRL_BMPDEC_get_formats ( formats );
                if ( SVG_NO_ERROR != ret_err )
                {
                    GRL_BMPDEC_set_error( ctx, ret_err );
                }
            }
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( ctx, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGGETBMPDECFORMATS");
    }

    SVG_BMP_U("EXIT FROM SVGGETBMPDECFORMATS");
}


void svgDecodeImage( SVGContextBmpDec    *ctx,
                     const SVGImage      *image )
{
    SVGError            ret_err         = SVG_NO_ERROR;
    S32                 cnt;
    GRL_BMPDEC_conf     config          = {0,0,0,0,0,0,0,0,0,0};
    SVGBoolean          is_valid        = SVG_FALSE;

    SVG_BMP_U("ENTER INTO SVGDECODEIMAGE");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        GRL_BMPDEC_get_conf( &config );

        /* Check params */
        if ( (ctx == NULL) || (image == NULL)  || (config.queue_size <= ctx->write_info_cnt) )
        {
            if (ctx == NULL)
            {
                /* Context NULL */
                ret_err = SVG_CONTEXT_NULL;
                GRL_BMPDEC_set_error( ctx, ret_err );
                SVG_BMP_E("SVG_CONTEXT_NULL IN SVGDECODEIMAGE");
            }
            else if (image == NULL)
            {
                /* Image pointer is NULL */
                ret_err = SVG_POINTER_NULL;
                GRL_BMPDEC_set_error( ctx, ret_err );
                SVG_BMP_E("SVG_POINTER_NULL IN SVGDECODEIMAGE");
            }
            else /* (SVG_DECODING_INFO_CNT <= bmp_ctx->write_info_cnt) */
            {
                /* Maximum number of queuable requests reached */
                ret_err = SVG_BMPDEC_QUEUE_MAX;
                GRL_BMPDEC_set_error ( ctx, ret_err );
                SVG_BMP_E("SVG_BMPDEC_QUEUE_MAX IN SVGDECODEIMAGE");
            }

        }
        else
        {
            /* Only handle valid contexts */
            is_valid = GRL_BMPDEC_clst_is_valid( ctx );

            if ( SVG_FALSE == is_valid )
            {
                ret_err = SVG_BMPDEC_INVALID_CONTEXT;
                SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN SVGDECODEIMAGE");
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
            else
            {
                /* Check the image structure */
                ret_err = GRL_BMPDEC_check_img( image );
                if ( SVG_NO_ERROR != ret_err )
                {
                    GRL_BMPDEC_set_error( ctx, ret_err );
                }
                else
                {
                    /* Set up decoding structure for queue */
                    cnt = ctx->write_info_cnt;
                    memcpy( &(ctx->p_write_queue[cnt].image), image, sizeof(SVGImage) );
                    ctx->p_write_queue[cnt].p_bmp_ctx = ctx;
                    ctx->write_info_cnt++;
                }
            }
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( ctx, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGDECODEIMAGE");
    }
    
    SVG_BMP_U("EXIT FROM SVGDECODEIMAGE");
}


void svgFlushBmpDec( SVGContextBmpDec   *ctx )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    SVGBoolean  is_valid        = SVG_FALSE;

    SVG_BMP_U("ENTER INTO SVGFLUSHBMPDEC");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        /* Check params */
        if ( ctx == NULL )
        {
            /* Context is NULL */
            GRL_BMPDEC_set_error( ctx, SVG_CONTEXT_NULL );
        }
        else
        {
            /* Only handle valid contexts */
            is_valid = GRL_BMPDEC_clst_is_valid( ctx );
            
            if ( SVG_FALSE == is_valid )
            {
                ret_err = SVG_BMPDEC_INVALID_CONTEXT;
                SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN SVGFLUSHBMPDEC");
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
            else
            {
                if ( ctx->write_info_cnt > 0 )
                {
                    ret_err = GRL_BMPDEC_draw_image_async( ctx );
                    if ( ret_err != GRL_NO_ERROR )
                    {
                        GRL_BMPDEC_set_error( ctx, ret_err );
                    }
                }
                else
                {
                    /* There are no requests to be processed */
                    ret_err = SVG_BMPDEC_QUEUE_EMPTY;
                    GRL_BMPDEC_set_error( ctx, ret_err );
                    SVG_BMP_E("SVG_BMPDEC_QUEUE_EMPTY IN SVGFLUSHBMPDEC");
                }
            }
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( ctx, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGFLUSHBMPDEC");
    }

    SVG_BMP_U("EXIT FROM SVGFLUSHBMPDEC");
}


void svgFinishBmpDec( SVGContextBmpDec  *ctx )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    SVGBoolean  is_valid        = SVG_FALSE;
    
    SVG_BMP_U("ENTER INTO SVGFINISHBMPDEC");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        /* Check params */
        if ( ctx == NULL )
        {
            /* Context is NULL */
            ret_err = SVG_CONTEXT_NULL;
            GRL_BMPDEC_set_error( ctx, ret_err );
            SVG_BMP_E("SVG_CONTEXT_NULL IN SVGFINISHBMPDEC");
        }
        else
        {
            /* Only handle valid contexts */
            is_valid = GRL_BMPDEC_clst_is_valid( ctx );
            
            if ( SVG_FALSE == is_valid )
            {
                ret_err = SVG_BMPDEC_INVALID_CONTEXT;
                SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN SVGFINISHBMPDEC");
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
            else
            {
                if ( ctx->write_info_cnt > 0 )
                {
                    ret_err = GRL_BMPDEC_draw_image_sync( ctx );

                    if ( ret_err != GRL_NO_ERROR )
                    {
                        GRL_BMPDEC_set_error( ctx, ret_err );
                    }
                }
                else
                {
                    /* There are no requests to be processed,
                     * but maybe the decoding task is working, wait for them*/
                	ret_err = GRL_BMPDEC_wai_dec( ctx );
					if ( SVG_NO_ERROR != ret_err )
					{
						GRL_BMPDEC_set_error( NULL, ret_err );
					}
                    ret_err = SVG_BMPDEC_QUEUE_EMPTY;
                    GRL_BMPDEC_set_error( ctx, ret_err );
                    SVG_BMP_E("SVG_BMPDEC_QUEUE_EMPTY IN SVGFINISHBMPDEC");
                }
            }
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( ctx, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGFINISHBMPDEC");
    }
    
    SVG_BMP_U("EXIT FROM SVGFINISHBMPDEC");
}


void svgWaitBmpDec( SVGContextBmpDec   *ctx )
{
    SVGError    ret_err         = SVG_NO_ERROR;
    SVGBoolean  is_valid        = SVG_FALSE;

    SVG_BMP_U("ENTER INTO SVGWAITBMPDEC");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        /* Check params */
        if ( ctx == NULL )
        {
            /* Context is NULL */
            GRL_BMPDEC_set_error( ctx, SVG_CONTEXT_NULL );
        }
        else
        {   
            /* Only wait for valid contexts */
            is_valid = GRL_BMPDEC_clst_is_valid( ctx );
            
            if ( SVG_FALSE == is_valid )
            {
                ret_err = SVG_BMPDEC_INVALID_CONTEXT;
                SVG_BMP_E("SVG_BMPDEC_INVALID_CONTEXT IN SVGWAITBMPDEC");
                GRL_BMPDEC_set_error( NULL, ret_err );
            }
            else
            {
                ret_err = GRL_BMPDEC_wai_dec( ctx );
                if ( SVG_NO_ERROR != ret_err )
                {
                    GRL_BMPDEC_set_error( NULL, ret_err );
                }
            }
        }
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( ctx, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGWAITBMPDEC");
    }

    SVG_BMP_U("EXIT FROM SVGWAITBMPDEC");
}


SVGError svgGetBmpDecError( SVGContextBmpDec *ctx )
{
    SVGError    ret_err         = SVG_NO_ERROR;

    SVG_BMP_U("ENTER INTO SVGGETBMPDECERROR");

    if ( SVG_TRUE == g_bmpdec_initialized )
    {
        ret_err = GRL_BMPDEC_get_error( ctx );
    }
    else
    {
        ret_err = SVG_NOT_INITIALIZED;
        GRL_BMPDEC_set_error( ctx, ret_err );
        SVG_BMP_E("SVG_NOT_INITIALIZED IN SVGGETBMPDECERROR");
    }

    SVG_BMP_U("EXIT FROM SVGGETBMPDECERROR");

    return ret_err;
}

